function [File] = OpenFiles(Settings);

% OPENFILES opens the images required for cell traction computation.
% Three types of images need to be opened:
%
% FLUORESCENCE IMAGE: this is the image of the fluorescent beads embedded in the gel 
% when the cell is on top of the gel. There may be more than one  
% fluorescence image in the case of time lapse experiments. Fluorescence images should be labeled as
% 'basename1.imgfmt', 'basename2.imgfmt', ..., 'basenameN.imgfmt'
%
% PHASE CONTRAST IMAGE: this is the phase contrast (or bright field or DIC) image of the cell
% It is used to draw the cell boundary to compute the constrained tractions. There may be more
% than one phase contrast image in the case of time lapse experiments. Phase Contrast images should be 
% labeled as 'basename1.imgfmt', 'basename2.imgfmt', ..., 'basenameN.imgfmt'
%
% TRYPSIN IMAGE: this is the image of the flourescent beads embedded in the gel when the gel is relaxed,
% usually after trypsinizing the cell. This is also called reference image
%
% INPUTS:
% Settings is a structure that contains at least the following fields
%       Imgfmt : format of the images ('tif', 'bmp', etc)
%
% OUTPUTS
% File: file is a structure that contains at least the following fields
%       pathname: path of the folder that contains the images
%       Base.Fluorescence: base of the fluorescence names, i.e., the name without number and extension
%       Base.PC: base of the phase contrast names, i.e., the name without number and extension
%       Base.Trypsin: base of the phase contrast names, i.e., the name without number and extension
%       File.NFiles.Fluorescence: number of fluorescence files in the folder
%       File.NFiles.PC: number of phase contrast files in the folder
%       File.NFiles.Trypsin: number of trypsin files in the folder
%       File.Name.Fluorescence: name of each fluorescence image
%       File.Name.PC: name of each PC image
%       File.Name.Trypsin: name of each trypsin
%
% Three folders are created:
%
% CROPPEDDATA: contains the images cropped so that the shift between the
% fluorescence image and the trypsin image is less than a pixel
%
% TRACTIONS: contains traction results
%
% DISPLACEMENTS: contains the displacement results
%
% by Xavier Trepat January 2008

% open the first fluorescence file of the series

% Edited by Md. Mydul Islam Miku, 2017 2018

[Temp.FluorescenceName, File.pathname] = uigetfile(['*.' Settings.Imgfmt], 'Open first fluorescence image');
cd(File.pathname);

% open the first phase contrast file
[Temp.PCName, File.pathname] = uigetfile(['*.' Settings.Imgfmt], 'Open first phase contrast image');

% open the reference file (image of the gel relaxed after trypsin)
[Temp.TrypsinName, File.pathname] = uigetfile(['*.' Settings.Imgfmt], 'Open reference image (image after trypsin)');
    
% Establish the bases of the name.
File.Base.Fluorescence = Temp.FluorescenceName(1:end-5); 
File.Base.PC = Temp.PCName(1:end-5);
File.Base.Trypsin = Temp.TrypsinName(1:end-5);  


% How many images form the fluorescence stack?
dirTrac = dir; 
imNum = 0;     
for i = 3:length(dirTrac)  
    if ~isempty(findstr(dirTrac(i).name, File.Base.Fluorescence)) 
        imNum = imNum + 1;              
    end
end
File.NFiles.Fluorescence = imNum;

% Build the names of the fluorescent files
for i=1:File.NFiles.Fluorescence
        prefix = [];
        File.Name(i).Fluorescence = strcat(File.pathname,File.Base.Fluorescence, num2str(prefix), num2str(i),'.',Settings.Imgfmt);
end

% How many images form the phase contrast stack?
dirTrac = dir;
imNum = 0;
for i = 3:length(dirTrac)
    if ~isempty(findstr(dirTrac(i).name, File.Base.PC))
        imNum = imNum + 1;                
    end
end
File.NFiles.PC = imNum;

% Build the names of the phase files
for i=1:File.NFiles.PC
        prefix = [];
        File.Name(i).PC = strcat(File.pathname,File.Base.PC, num2str(prefix), num2str(i),'.',Settings.Imgfmt);
end

% Build the names of the fluorescent files
File.Name(1).Trypsin = strcat(File.pathname,File.Base.Trypsin, num2str(prefix), num2str(1),'.',Settings.Imgfmt);

% Create folders
if exist('croppeddata') ~= 7
    mkdir('croppeddata');
end

if exist('Tractions') ~= 7
    mkdir('Tractions');
end

if exist('Displacements') ~= 7
    mkdir('Displacements');
end
%File.NFiles.Fluorescence = imNum-1;
disp(File.NFiles.Fluorescence)